



/***********************************************************************/
/* INCLUDES                                                            */
/***********************************************************************/
#include "ntag_i2c.h"
#include "fsl_common.h"


/***********************************************************************/
/* DEFINES                                                             */
/***********************************************************************/




/*! *********************************************************************************
* \brief  Initialize the NTAG I2C as master role.
*
* \param[in\out] - none
*
* \return - none
********************************************************************************** */
void NTAG_I2cInitDevice(void)
{
    /* Reset */
    SYSCON->RST_SW_SET = (0x01U << 18);
    SYSCON->RST_SW_CLR = (0x01U << 18);
    /* Enable clock */
    SYSCON->CLK_EN = (0x01U << 18);
    
    /* Wait for not busy */
    while( NTAG_I2C->SR & (0x01U << 1));
    /* Enable I2C, SCL = 16MHz / (2 ^ (2 + 2)) / (1 + 1) / 5 = 200K */
    NTAG_I2C->CR = (0x02U << 30) | (0x1U << 24) | (0x0U << 10) | (0x01U << 8);
}


/*! *********************************************************************************
* \brief  Enumerate to find a valid NTAG's I2C slave(NTAG) device 
* 
*					Any read to the NTAG's I2C address 
*			 		will only return 0x04 which is manufacturer ID for NXP Semiconductors.
*			 		For the master who want to access the NTAG device enumeration is needed.
*
* \param[in] none
*
* \return - NTAG's I2C address 
********************************************************************************** */
uint8_t NTAG_Enumeration(void)
{
    uint8_t slave_addr = 0;
    for (int i = 1; i < 0x80; i++)
    {
        /* Wait for not busy */
        while( NTAG_I2C->SR & (0x01U << 1));

        /* Send start with write direction */
        NTAG_I2C->TXD = (0x01U << 16) | (i << 1);
        /* Wait for Tx interrupt */
        while(!(NTAG_I2C->INT & 0x01U));
        /* Check ACK */
        if(!(NTAG_I2C->SR & 0x01U))
        {
            /* ACK received */
            slave_addr = i;
        }
        /* Send stop */
        NTAG_I2C->TXD = (0x01U << 17);
        /* Clear tx interrupt */
        NTAG_I2C->INT = 0x01;
        
        for (int j = 0; j < 100000; j++);

        if ((slave_addr == i) && (i != 0))
        {
            break;
        }
    }
		return slave_addr;
}



/*! *********************************************************************************
* \brief send n byte to NTAG via I2C
*
*			 	 For NTAG write operation it should always be in the unit of 16 byte(one block).  			 	
*
* \param[in] address 	- I2C address 
*
* \param[in] *p_data 	- Point to address of the data to be writen to NTAG
*
* \param[in] size 		- The length of data 
*
* \return NTAG_Status_t - gNtagSuccess_c or error code
********************************************************************************** */
NTAG_Status_t NTAG_MemoryWrite(uint8_t address, uint8_t *p_data, uint8_t size)
{
    uint8_t idx = 0;
		//uint8_t len;
    NTAG_Status_t error = gNtagSuccess_c;

    /* Wait for not busy */
    while( NTAG_I2C->SR & (0x01U << 1));
		
    /* Send start with write direction */
    NTAG_I2C->TXD = (0x01U << 16) | (address << 1);
		
		idx = 0;
    while(idx < size)
    {
        /* Wait for Tx interrupt */
        while(!(NTAG_I2C->INT & 0x01U));
        /* Check ACK */
        if(NTAG_I2C->SR & 0x01U)
        {
            /* NACK received */
					  error = gNtagNACKReceive_c;
					
						/* issue stop */
						NTAG_I2C->TXD = (0x01U << 17);
						/* Clear tx interrupt */
						NTAG_I2C->INT = 0x01;
					
            return error;
        }

        /* Send data that consists of memory and 16byte data*/
        NTAG_I2C->TXD = (0x01U << 18) | p_data[idx++];

        /* Clear tx interrupt */
        NTAG_I2C->INT = 0x01;
    }

    /* Wait for Tx interrupt */
    while(!(NTAG_I2C->INT & 0x01U));
		 /* Check ACK */
		if(NTAG_I2C->SR & 0x01U)
    {
				/* NACK received */
				error = gNtagNACKReceive_c;
		}
    /* issue stop */
    NTAG_I2C->TXD = (0x01U << 17);
    /* Clear tx interrupt */
    NTAG_I2C->INT = 0x01;

    return error;
}

/*! *********************************************************************************
* \brief  read n byte from NTAG I2C
*
*			 For NTAG read operation it shoud always be in the unit of 16 byte(one block).
*			 Less than 16 bytes read will cause I2C busy until power cycle the system.
*
* \param[in] address 	- I2C block address 
*
* \param[out] *p_data - Point to address of the data to be stored in host.
*
* \param[in] size 		- The length of data to be read
*
* \return NTAG_Status_t - gNtagSuccess_c or error code
********************************************************************************** */
NTAG_Status_t NTAG_MemoryRead(uint8_t address, uint8_t *p_data, uint8_t size)
{
		uint8_t i;//len
    NTAG_Status_t error = gNtagSuccess_c;

    /* Wait for not busy */
    while( NTAG_I2C->SR & (0x01U << 1));

    /* Send re-start with read direction */
    NTAG_I2C->TXD = (0x01U << 16) | (address << 1) | 0x01U;
	
    /* Wait for Tx interrupt */
    while(!(NTAG_I2C->INT & 0x01U));
	
    /* Check ACK */
    if(NTAG_I2C->SR & 0x01U)
    {
        /* NACK received */
        error = gNtagNACKReceive_c;
				/* issue stop */
				NTAG_I2C->TXD = (0x01U << 17);
				/* Clear tx interrupt */
				NTAG_I2C->INT = 0x01;

			  return error;
    }

    for (i = 0; i < size; i++)
    {
        /* Prepare ACK and select read */
        NTAG_I2C->TXD = (0x01U << 19) | (0x00U << 20);
        /* Clear Rx or Tx interrupt */
        NTAG_I2C->INT = 0x03;
        /* Wait for Rx interrupt */
        while(!(NTAG_I2C->INT & 0x02U));
        p_data[i] = NTAG_I2C->RXD;
    }
    
    /* Stop */
    NTAG_I2C->TXD = (0x01U << 17);
    /* Clear Rx interrupt */
    NTAG_I2C->INT = 0x02;

    return error;
}



